/*           Bobby Fischer-Syle Chess Clock     Version 2.0

          Written by Phillip M. Feldman and Michael J. Feldman

                           20 Sept., 2008


OVERVIEW

This Lite-C program implements a Bobby Fischer-style chess clock.  Each
player has an initial allocation of time, and is granted a small increment
of time following a move.


CONTROLS

After launching the program, you will be prompted to enter the initial time
allocation in minutes and the per-move time increment in seconds.  The
default values for both of these are 10 (but note that the time units are
different).  These values must be the same for both players, i.e., there is
currently no mechanism for granting one player a time handicap.

After these two values have been specified, a press of the spacebar starts
player #1's clock counting down in half-second increments.  Any subsequent
press of the spacebar toggles the active clock, i.e., switches between
clock #1 and clock #2.

You may press 'esc' at any time to end the program.

Press 'p' to pause the clock and 'r' to resume.  While the clocks are
paused, no keys other than 'r' and 'esc' have any effect.


REFERENCES

1. http://en.wikipedia.org/wiki/Game_clock

2. http://www.conitec.com/english/gstudio/litec.htm


REVISION HISTORY

Version 1.2, Aug. 14, 2008: Modified code to replace PANEL object with
two separate TEXT objects for displaying the user times.  The reason for
this change is that with the PANEL object, we cannot independently set
the colors of the two displayed times.  Added code to display the time in
red when the values falls below 60 seconds.
*/

#include <acknex.h>
#include <default.c>

/* The variable "Mode" equals 0 until the user(s) start the clock running
by pressing the space bar, at which time the value of Mode is changed to
1.  Subsequent space bar presses toggle Mode between 1 and 2. */

var Mode= 0;

// t_1 and t_2 store the times for players 1 and 2, respectively:

var t_1, t_1_min, t_1_sec, t_2, t_2_min, t_2_sec, t_extra, t_warn= 59.75;

TEXT* title= {
  pos_x = 375; pos_y = 50;
  font = "Arial#32bi";
  string("Bobby Fischer-style Chess Clock",
   "Written by", "Phillip M. Feldman and Michael J. Feldman");
  flags= CENTER_X | VISIBLE;
}

TEXT* help= {
   pos_x = 100; pos_y = 200;
   font = "Arial#24bi";
   flags= VISIBLE;
   strings= 1;
}

STRING* user_input_str= "            "; // initialized to 12 blanks

TEXT* user_input_txt= {
  pos_x = 100; pos_y = 240;
  font = "Arial#24bi";
  string(user_input_str);
  flags= VISIBLE;
}

// Player #1's displayed clock time:

TEXT* time_txt_1= {
  pos_x = 100; pos_y = 300;
  font = "Arial#60b";
  flags= VISIBLE;
  strings= 2;
}

// Player #2's displayed clock time:

TEXT* time_txt_2= {
  pos_x = 450; pos_y = 300;
  font = "Arial#60b";
  flags= VISIBLE;
  strings= 2;
}

SOUND* beep= "chord.wav";
var sound_handle;
sound_handle = snd_play(beep, 100, 0);


// ------------------------------------------------------------------------


/* The following function handles space bar presses.  The initial space bar
bar press changes the value of "Mode" from 0 to 1.  Subsequent space bar
presses toggle the value of Mode between 1 and 2. */

function spacebar_press() {

   switch(Mode) {

      // Mode == 0 means that play has not yet started, so start it now:
      case 0:
      Mode= 1;
      // Update help text:
      str_cpy( (help.pstring)[0],
        "Press the spacebar to switch clocks or press 'p' to pause.");
      break;

      // Mode == 1 means that it is player #1's turn and that the game is
      // not currently paused:
      case 1:
      // Add t_extra seconds to player #1's time:
      t_1+= t_extra;

      // Update displayed time on screen:
      t_1_min= integer(t_1/60.0); t_1_sec= abs(t_1 - 60.0 * t_1_min);

      str_cpy(     (time_txt_1.pstring)[0], "Player 1: ");
      str_cpy(     (time_txt_1.pstring)[1], "");
      str_cat_num( (time_txt_1.pstring)[1], "%3.0f : ", t_1_min);
      str_cat_num( (time_txt_1.pstring)[1], "%02.1f", t_1_sec);

      /* If player #1's time is greater than t_warn (possibly because of
      the extra time), set text color to white. */
      if (t_1 >= t_warn) {
         time_txt_1.red= 255; time_txt_1.green= 255; time_txt_1.blue= 255;
      }
      Mode= 2;
      break;

      // Mode == 2 means that it is player #2's turn and that the game is
      // not currently paused:
      case 2:
      // Add t_extra seconds to player #2's time:
      t_2+= t_extra;

      // Update displayed time on screen:
      t_2_min= integer(t_2/60.0); t_2_sec= abs(t_2 - 60.0 * t_2_min);

      str_cpy(     (time_txt_2.pstring)[0], "Player 2: ");
      str_cpy(     (time_txt_2.pstring)[1], "");
      str_cat_num( (time_txt_2.pstring)[1], "%3.0f : ", t_2_min);
      str_cat_num( (time_txt_2.pstring)[1], "%02.1f", t_2_sec);

      /* If player #2's time is greater than t_warn (possibly because of
      the extra time), set text color to white. */
      if (t_2 >= t_warn) {
         time_txt_2.red= 255; time_txt_2.green= 255; time_txt_2.blue= 255;
      }
      Mode= 1;
      break;

      // Use of the space bar is not allowed when Mode == 3 or Mode == 4:
      default:
      sound_handle= snd_play(beep, 100, 0);

   }

} // function spacebar_press()


// The following function handles the key "p", which pauses the clock.

function p_press() {

   switch(Mode) {

      // Mode == 1 means that it is player #1's turn and that the clock is
      // not currently paused:
      case 1:
      // Update help text:
      str_cpy( (help.pstring)[0], "Press 'r' to resume.");
      Mode= 3;
      break;

      // Mode == 2 means that it is player #2's turn and that the clock is
      // not currently paused:
      case 2:
      // Update help text:
      str_cpy( (help.pstring)[0], "Press 'r' to resume.");
      Mode= 4;
      break;

      // If Mode equals anything else, the clock is already paused:
      default:
      sound_handle= snd_play(beep, 100, 0);

   }

} // function p_press()


// The following function handles the key "r", which resumes the clock if
// it is currently paused:

function r_press() {

   switch(Mode) {

      // Mode == 3 means that it is player #1's turn and that the clock is
      // currently paused:
      case 3:
      Mode= 1;
      // Update help text:
      str_cpy( (help.pstring)[0],
        "Press the spacebar to switch clocks or press 'p' to pause.");
      break;

      // Mode == 4 means that it is player #2's turn and that the clock is
      // currently paused:
      case 4:
      Mode= 2;
      // Update help text:
      str_cpy( (help.pstring)[0],
        "Press the spacebar to switch clocks or press 'p' to pause.");
      break;

      // If Mode equals anything else, the timer is not currently paused,
      // and the resume command thus does not make sense.
      default:
      sound_handle= snd_play(beep, 100, 0);

   }

} // function r_press()


// ------------------------------------------------------------------------


function main() {


   // Section 1: Set screen mode and background color.

   video_mode = 7; // create a program window of 800x600 pixels
   screen_color.blue = 50; // and make its background dark blue


   /* Section 2: Prompt the user to specify initial time.  (This code is
   based on the example in http://www.conitec.net/beta/ainkey.htm). */

   // Display prompt #1:
   str_cpy( (help.pstring)[0],
     "Enter initial time in minutes (default= 10): ");

   // Wait for user to type something (or nothing) followed by Enter:
   inkey(user_input_str);

   // Convert user input string to a number and convert number to seconds:
   t_2= t_1= 60 * str_to_num(user_input_str);

   if (t_1 <= 0) {
      t_2= t_1= 600; // Assign default value.
   }

   // Display initial times:

   t_1_min= integer(t_1/60.0); t_1_sec= abs(t_1 - 60.0 * t_1_min);
   str_cpy(     (time_txt_1.pstring)[0], "Player 1: ");
   str_cpy(     (time_txt_1.pstring)[1], "");
   str_cat_num( (time_txt_1.pstring)[1], "%3.0f : ", t_1_min);
   str_cat_num( (time_txt_1.pstring)[1], "%02.1f", t_1_sec);

   t_2_min= integer(t_2/60.0); t_2_sec= abs(t_2 - 60.0 * t_2_min);
   str_cpy(     (time_txt_2.pstring)[0], "Player 2: ");
   str_cpy(     (time_txt_2.pstring)[1], "");
   str_cat_num( (time_txt_2.pstring)[1], "%3.0f : ", t_2_min);
   str_cat_num( (time_txt_2.pstring)[1], "%02.1f", t_2_sec);


   // Display prompt #2:
   str_cpy( (help.pstring)[0],
     "Enter extra time per move in seconds (default= 10): ");

   // Wipe out contents of user input string, replacing with 12 blanks:
   str_cpy(user_input_str, "            ");

   // Wait for user to type something (or nothing) followed by Enter:
   inkey(user_input_str);

   // Convert user input string to a number:
   t_extra= str_to_num(user_input_str);

   if (t_extra <= 0) {
      t_extra= 10; // Assign default value.
   }


   /* Section 3: Display initial help text, define mappings from keys to
   functions, and then wait for user to press the space bar. */

   // Display initial help text:
   str_cpy( (help.pstring)[0], "Press the spacebar to start clock #1.");

   // The user presses the space bar to signal that a player has ended his
   // or her turn.  We could scan the key state in the main loop, but this
   // would "lock" the CPU, which is inefficient and inelegant.  A key
   // event provides a better mechanism.

   on_space= spacebar_press;

   on_p= p_press;
   on_r= r_press;


   while (1) {

      // Wait for half a second:
      wait(-0.5);


      // Player #1's turn:

      if (Mode == 1) {

         // Decrement player #1's time:
         t_1-= 0.5;

         // Update displayed time on screen:
         t_1_min= integer(t_1/60.0); t_1_sec= abs(t_1 - 60.0 * t_1_min);

         str_cpy(     (time_txt_1.pstring)[0], "Player 1: ");
         str_cpy(     (time_txt_1.pstring)[1], "");
         str_cat_num( (time_txt_1.pstring)[1], "%3.0f : ", t_1_min);
         str_cat_num( (time_txt_1.pstring)[1], "%02.1f", t_1_sec);

         // If less then t_warn seconds remaining, change text color to red:
         if (t_1 < t_warn) {
            time_txt_1.red= 255; time_txt_1.green= 0; time_txt_1.blue= 0;
         }

         // If out of time, terminate the program:
         if (t_1 <= 0) break;
      }


      // Player #2's turn:

      if (Mode == 2) {

         // Decrement player #2's time:
         t_2-= 0.5;

         // Update displayed time on screen:
         t_2_min= integer(t_2/60.0); t_2_sec= abs(t_2 - 60.0 * t_2_min);

         str_cpy(     (time_txt_2.pstring)[0], "Player 2: ");
         str_cpy(     (time_txt_2.pstring)[1], "");
         str_cat_num( (time_txt_2.pstring)[1], "%3.0f : ", t_2_min);
         str_cat_num( (time_txt_2.pstring)[1], "%02.1f", t_2_sec);

         // If less then t_warn seconds remaining, change text color to red:
         if (t_2 < t_warn) {
            time_txt_2.red= 255; time_txt_2.green= 0; time_txt_2.blue= 0;
         }

         // If out of time, terminate the program:
         if (t_2 <= 0) break;
      }

   }

} // function main
